/*:
 * @target MZ
 * @plugindesc v1.0 装備・スキルのメモ欄から戦闘開始時にステートを付与します。
 * @author ChatGPT
 *
 * @help
 * 装備やスキルのメモ欄に書かれたステートIDのステートを、
 * 戦闘開始時にのみ付与するプラグインです。
 *
 * ▼ メモ欄タグ
 *   <BattleStartState: 3>
 *   <BattleStartState: 4,5,6>
 *
 * ・アクター
 *   - 装備している武器・防具のメモ欄
 *   - アクターが習得しているスキルのメモ欄
 *
 * ・敵キャラ
 *   - 敵データ「行動」に設定されているスキルのメモ欄
 *
 * ※ 戦闘開始時に一度だけ addState します。
 * ※ ステートの解除タイミングは、ステートの設定に従います。
 *
 * このプラグインにプラグインコマンドやパラメータはありません。
 */

(() => {
    "use strict";

    /**
     * 指定オブジェクト（武器・防具・スキルなど）のメモ欄から
     * <BattleStartState: x,x,x> を読み取り、ステートID配列を返す
     */
    function battleStartStateIdsFromObject(obj) {
        const result = [];
        if (!obj || !obj.note) return result;

        const re = /<BattleStartState\s*:\s*([\d\s,]+)>/ig;
        let match;
        while ((match = re.exec(obj.note)) !== null) {
            const list = match[1].split(",");
            for (const item of list) {
                const id = Number(item.trim());
                if (id > 0) {
                    result.push(id);
                }
            }
        }
        return result;
    }

    /**
     * 重複を取り除いた配列を返す
     */
    function uniqueArray(arr) {
        return [...new Set(arr)];
    }

    /**
     * Game_Battler 基本実装
     */
    Game_Battler.prototype.getBattleStartStateIds = function() {
        // デフォルトは何も付与しない
        return [];
    };

    Game_Battler.prototype.applyBattleStartStates = function() {
        const ids = this.getBattleStartStateIds();
        for (const id of ids) {
            if (!this.isStateAffected(id)) {
                this.addState(id);
            }
        }
    };

    /**
     * Game_Actor: 装備 + 習得スキルから取得
     */
    const _Game_Actor_getBattleStartStateIds = Game_Actor.prototype.getBattleStartStateIds;
    Game_Actor.prototype.getBattleStartStateIds = function() {
        let result = [];
        // 装備
        for (const equip of this.equips()) {
            if (equip) {
                result = result.concat(battleStartStateIdsFromObject(equip));
            }
        }
        // スキル（習得済み）
        for (const skill of this.skills()) {
            if (skill) {
                result = result.concat(battleStartStateIdsFromObject(skill));
            }
        }
        // 必要なら元実装の結果も足す（将来の拡張用）
        if (_Game_Actor_getBattleStartStateIds !== Game_Actor.prototype.getBattleStartStateIds) {
            result = result.concat(_Game_Actor_getBattleStartStateIds.call(this));
        }
        return uniqueArray(result);
    };

    /**
     * Game_Enemy: 行動に設定されたスキルから取得
     */
    const _Game_Enemy_getBattleStartStateIds = Game_Enemy.prototype.getBattleStartStateIds;
    Game_Enemy.prototype.getBattleStartStateIds = function() {
        let result = [];
        const enemy = this.enemy();
        if (enemy && enemy.actions) {
            for (const action of enemy.actions) {
                const skill = $dataSkills[action.skillId];
                if (skill) {
                    result = result.concat(battleStartStateIdsFromObject(skill));
                }
            }
        }
        if (_Game_Enemy_getBattleStartStateIds !== Game_Enemy.prototype.getBattleStartStateIds) {
            result = result.concat(_Game_Enemy_getBattleStartStateIds.call(this));
        }
        return uniqueArray(result);
    };

    /**
     * BattleManager.startBattle をフックして、
     * 戦闘開始時に全バトラーへステートを付与
     */
    const _BattleManager_startBattle = BattleManager.startBattle;
    BattleManager.startBattle = function() {
        _BattleManager_startBattle.call(this);
        this.applyBattleStartStatesToAll();
    };

    BattleManager.applyBattleStartStatesToAll = function() {
        const allBattlers = [].concat($gameParty.members(), $gameTroop.members());
        for (const battler of allBattlers) {
            if (battler && battler.applyBattleStartStates) {
                battler.applyBattleStartStates();
            }
        }
    };

})();
